/* -*- C++ -*- */

/*
    Copyright (C) 2011 William Brodie-Tyrrell
    william@brodie-tyrrell.org
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of   
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>
*/


#ifndef _FSTOP_TIMER_H_
#define _FSTOP_TIMER_H_

#include "WProgram.h"
#include <LiquidCrystal.h>
#include <EEPROM.h>
#include <Keypad.h>

// EEPROM layout
#define EE_BACKLIGHT 0x00
#define EE_DRYDOWN 0x01
#define EE_DRYAPPLY 0x02
#define EE_STRIPBASE 0x10
#define EE_STRIPSTEP 0x12
#define EE_STRIPCOV 0x14

/**
 * Definition of a program of exposures
 */
class Program {
    static const int INVALID=0x0000;
    static const int SLOTBITS=7;
    static const int SLOTBASE=0x80;
    static const int TEXTLEN=14;
    static const long MAXMS=999999L;    // ceiling of 1000s
  
public:

    static const int MAXEXP=8;
    static const int FIRSTSLOT=1;
    static const int LASTSLOT=7;

    /// a single exposure step
    class Exposure {
    public:
  
        /// render the exposure settings to the screen (time and text)
        /// @param lin also display linear time (millis)
        void display(LiquidCrystal &disp, char *buf, bool lin);
        /// rended only the time line (bottom row);
        void displayTime(LiquidCrystal &disp, char *buf, bool lin);
  
        int stops;        // fixed-point, 1/100ths of a stop
        unsigned long ms; // milliseconds to expose (post-compilation, not saved)
        char text[TEXTLEN+1];    // description (only 14 bytes written to EEPROM)
    };

    /// clear all entries, leaving base exposure of 0 stops
    void clear();
  
    /// step accessor
    Exposure &operator[](int which);
  
    /// determine number of valid exposure objects; first = base
    // unsigned char getCount();
  
    /// convert a program from stops to linear time so that it can be execed
    bool compile(char dd);
  
    /// save to EEPROM
    /// @param slot slot-number in 1..7
    void save(int slot);
  
    /// load from EEPROM
    /// @param slot slot-number in 1..7
    void load(int slot);
  
    /// configure the program as a test strip;
    /// is assumed to compile after this.
    void configureStrip(int base, int step, bool cover);

private:

    void compileStripIndiv(char dd);
    void compileStripCover(char dd);
    bool compileNormal(char dd);
    void clipExposures();

    /// convert hundredths-of-stops to milliseconds
    static unsigned long hunToMillis(int hunst);

    // compilation settings
    bool isstrip, cover;

    /// first exposure is base, rest as dodges/burns
    Exposure exposures[MAXEXP];
};

class Executor {
public:
    Executor(LiquidCrystal &d, Keypad &k, char p_e);

    void begin();

    /// must call this to define what will be execed
    /// @param p the current program
    void setProgram(Program *p);

    /// set drydown indication
    /// @param d whether to indicate that drydown is applied
    void setDrydown(bool d);

    Program *getProgram() const { return current; }

    /// define which exposure step to do
    void changePhase(unsigned char ph);
    unsigned char getPhase() const { return execphase; }

    /// move onto next phase
    void nextPhase();

    /// do a controlled exposure
    void expose();
  
private:    

    /// program we're working on
    Program *current;
  
    LiquidCrystal &disp;
    Keypad &keys;
    char pin_expose;
    char dispbuf[17];

    bool dd;

    /// program phase about to be executed
    unsigned char execphase;
};

/**
 * State-machine implementing fstop timer
 *
 */
class FstopTimer {
public:

    enum {
        ST_MAIN,
        ST_EDIT,
        ST_EDIT_EV,
        ST_EDIT_TEXT,
        ST_EXEC,
        ST_FOCUS,
        ST_IO,
        ST_IO_LOAD,
        ST_IO_SAVE,
        ST_TEST,
        ST_TEST_CHANGEB,
        ST_TEST_CHANGES,
        ST_CONFIG,
        ST_CONFIG_DRY,
        ST_COUNT
    };

    /// constructor
    /// @param l display to write to
    /// @param k keypad to read from
    /// @param p_e exposure pin (high = on)
    /// @param p_eb expose-button pin; currently unused!
    /// @param p_p beep pin (not used much)
    /// @param p_bl backlight pin, connect via BC548
    FstopTimer(LiquidCrystal &l, SMSKeypad &k, 
               char p_e, char p_eb, char p_b, char p_bi);

    /// setup IO
    void begin();

    /// continue exucution of current state, ponder input
    void poll();

private:

    char inbuf[17];
    char dispbuf[17];
    
    LiquidCrystal &disp;
    SMSKeypad &keys;
    SMSKeypad::Context smsctx;
    DecimalKeypad deckey;
    DecimalKeypad::Context expctx;
    DecimalKeypad::Context stepctx;
    DecimalKeypad::Context dryctx;
    DecimalKeypad::Context intctx;

    /// programs to execute
    Program current, strip;
    /// current config for test strips
    int stripbase, stripstep;
    bool stripcover;

    Executor exec;

    /// LCD PWM factor
    int brightness;
    /// whether drydown correction is currently applied
    bool drydown_apply;
    /// drydown factor
    char drydown; 
    /// exposure that is being edited
    int expnum;

    /// hardware pin numbers
    char pin_expose, pin_exposebtn, pin_beep, pin_backlight;

    /// all state-machine functions have this signature
    typedef void (FstopTimer::* voidfunc)();
    static voidfunc sm_enter[ST_COUNT];
    static voidfunc sm_poll[ST_COUNT];

    /// current state
    int curstate;
    int prevstate;

    void errorBeep();

    /**
     * Enter a new state in the state machine
     */
    void changeState(int st);

    /// invert and save the drydown-application bit
    void toggleDrydown();

    /// exec the current program
    void execCurrent();

    /// exec the test strip
    void execTest();

    void st_main_enter();
    void st_main_poll();
    void st_exec_enter();
    void st_exec_poll();
    void st_focus_enter();
    void st_focus_poll();
    void st_edit_enter();
    void st_edit_poll();
    void st_edit_ev_enter();
    void st_edit_ev_poll();
    void st_edit_text_enter();
    void st_edit_text_poll();
    void st_io_enter();
    void st_io_poll();
    void st_io_load_enter();
    void st_io_load_poll();
    void st_io_save_enter();
    void st_io_save_poll();
    void st_test_enter();
    void st_test_poll();
    void st_test_changeb_enter();
    void st_test_changeb_poll();
    void st_test_changes_enter();
    void st_test_changes_poll();
    void st_config_enter();
    void st_config_poll();
    void st_config_dry_enter();
    void st_config_dry_poll();

    // backlight bounds
    static const char BL_MIN=2;
    static const char BL_MAX=8;


};


#endif // _FSTOP_TIMER_H_
