/*
    Copyright (C) 2011 William Brodie-Tyrrell
    william@brodie-tyrrell.org
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of   
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>
*/


#include <LiquidCrystal.h>
#include <EEPROM.h>
#include <Keypad.h>
#include <FstopTimer.h>

/// functions to exec on entering each state
FstopTimer::voidfunc FstopTimer::sm_enter[]
   ={ &FstopTimer::st_main_enter,
      &FstopTimer::st_edit_enter, 
      &FstopTimer::st_edit_ev_enter, 
      &FstopTimer::st_edit_text_enter, 
      &FstopTimer::st_exec_enter,
      &FstopTimer::st_focus_enter,
      &FstopTimer::st_io_enter, 
      &FstopTimer::st_io_load_enter, 
      &FstopTimer::st_io_save_enter, 
      &FstopTimer::st_test_enter, 
      &FstopTimer::st_test_changeb_enter, 
      &FstopTimer::st_test_changes_enter, 
      &FstopTimer::st_config_enter,
      &FstopTimer::st_config_dry_enter };
/// functions to exec when polling within each state
FstopTimer::voidfunc FstopTimer::sm_poll[]
   ={ &FstopTimer::st_main_poll, 
      &FstopTimer::st_edit_poll,
      &FstopTimer::st_edit_ev_poll,
      &FstopTimer::st_edit_text_poll,
      &FstopTimer::st_exec_poll, 
      &FstopTimer::st_focus_poll, 
      &FstopTimer::st_io_poll, 
      &FstopTimer::st_io_load_poll, 
      &FstopTimer::st_io_save_poll, 
      &FstopTimer::st_test_poll,
      &FstopTimer::st_test_changeb_poll,
      &FstopTimer::st_test_changes_poll,
      &FstopTimer::st_config_poll,
      &FstopTimer::st_config_dry_poll };


FstopTimer::FstopTimer(LiquidCrystal &l, SMSKeypad &k, 
                       char p_e, char p_eb, char p_b, char p_bl)
    : disp(l), keys(k), 
      smsctx(&inbuf[0], 14, &disp, 0, 0),
      deckey(keys),
      expctx(&inbuf[0], 1, 2, &disp, 0, 1, true),
      stepctx(&inbuf[0], 1, 2, &disp, 0, 1, false),
      dryctx(&inbuf[0], 0, 2, &disp, 0, 1, false),
      intctx(&inbuf[0], 1, 0, &disp, 0, 1, false),
      exec(l, keys, p_e),
      pin_expose(p_e), pin_exposebtn(p_eb),
      pin_beep(p_b), pin_backlight(p_bl)
{

    // init libraries
    prevstate=curstate=ST_MAIN;
}

void FstopTimer::begin()
{
    // init raw IO
    pinMode(pin_expose, OUTPUT);
    pinMode(pin_beep, OUTPUT);
    pinMode(pin_backlight, OUTPUT);
    pinMode(pin_exposebtn, INPUT);

    digitalWrite(pin_expose, LOW);

    // load & apply backlight settings
    brightness=EEPROM.read(EE_BACKLIGHT);
    if(brightness > BL_MAX)
        brightness=BL_MAX;
    if(brightness < BL_MIN)
        brightness=BL_MIN;
    analogWrite(pin_backlight, (1<<brightness)-1);
   
    drydown=EEPROM.read(EE_DRYDOWN);
    drydown_apply=EEPROM.read(EE_DRYAPPLY);

    current.clear();
    stripbase=(EEPROM.read(EE_STRIPBASE)<<8) | EEPROM.read(EE_STRIPBASE+1);
    stripstep=(EEPROM.read(EE_STRIPSTEP)<<8) | EEPROM.read(EE_STRIPSTEP+1);
    stripcover=EEPROM.read(EE_STRIPCOV);

    disp.clear();
    disp.print("F/Stop Timer 0.2");
    disp.setCursor(0, 1);
    disp.print("W Brodie-Tyrrell");

    // wait
    keys.readBlocking();

    // boot the state machine
    changeState(ST_MAIN);

    exec.begin();
}

void FstopTimer::toggleDrydown()
{
    drydown_apply=!drydown_apply;
    EEPROM.write(EE_DRYAPPLY, drydown_apply);
}

/// begin the main-menu state; write to display
void FstopTimer::st_main_enter()
{
    disp.clear();
    disp.setCursor(0,0);
    disp.print("A:Edit   B: IO");
    disp.setCursor(0,1);
    disp.print("C:Config D: Test");
}

/// polling function executed while in the Main state
void FstopTimer::st_main_poll()
{   
    if(keys.available()){
        char ch=keys.readAscii();
        if(isspace(ch))
            return;
        switch(ch){
        case 'A':
            // exec.current.clear();
            changeState(ST_EDIT);
            break;
        case 'B':
            changeState(ST_IO);        
            break;
        case 'C':
            changeState(ST_CONFIG);
            break;
        case 'D':
            changeState(ST_TEST);
            break;
        case '#':
            exec.setProgram(&current);
            changeState(ST_EXEC);
            break;
        case '*':
            changeState(ST_FOCUS);
            break;
        default:
            errorBeep();
        }
    }
}

void FstopTimer::execCurrent()
{
    if(!current.compile(drydown_apply ? drydown : 0)){
        disp.print("Cannot Print");
        disp.setCursor(0, 1);
        disp.print("Dodges > Base");
        exec.setProgram(NULL);
        errorBeep();
        delay(2000);
        changeState(ST_EDIT);
    }
    else{
        exec.setProgram(&current);
        changeState(ST_EXEC);
    }
}

void FstopTimer::execTest()
{
    strip.configureStrip(stripbase, stripstep, stripcover);
    exec.setProgram(&strip);
    changeState(ST_EXEC);
}

void FstopTimer::st_exec_enter()
{
    Program *p=exec.getProgram();
    // we assume it compiles if we're in this state
    p->compile(drydown_apply ? drydown : 0);
    disp.clear();
    exec.setDrydown(drydown_apply);
}

void FstopTimer::st_exec_poll()
{
    if(keys.available()){
        char ch=keys.readAscii();
        if(isspace(ch))
            return;
        switch(ch){
        case '#':
            // perform exposure!
            exec.expose();
            break;
        case '*':
            // focus
            changeState(ST_FOCUS);
            break;
        case 'A':
            // abort/restart
            disp.clear();
            disp.print("Restart Exposure");
            delay(1000);
            exec.changePhase(0);
            break;
        case 'B':
            // skip exposure
            exec.nextPhase();
            break;
        case 'C':
            // main menu
            changeState(ST_MAIN);
            break;
        case 'D':
            // toggle drydown & recompile
            toggleDrydown();
            if(exec.getPhase() != 0){
                disp.clear();
                disp.print("Restart Exposure");
                disp.setCursor(0,1);
                disp.print("For Drydown Chg");
                delay(1000);
            }
            // forces recompile -> apply drydown
            changeState(ST_EXEC);
            break;
        } 
    }
}

void FstopTimer::st_focus_enter()
{
    disp.clear();
    disp.print("     Focus!");    
    digitalWrite(pin_expose, HIGH);
}

void FstopTimer::st_focus_poll()
{
    // return to prev state
    if(keys.available()){
        keys.readRaw();
        digitalWrite(pin_expose, LOW);
        changeState(prevstate);
    }
}

void FstopTimer::st_edit_enter()
{
    // don't print help
/*
    disp.clear();
    disp.setCursor(0,0);
    disp.print("Edit Program");
    disp.setCursor(0,1);
    disp.print("A:TX B:EV n:SLOT");
    delay(1000);
*/

    expnum=0;
    current[expnum].display(disp, dispbuf, false);
}

void FstopTimer::st_edit_poll()
{
    // also check for the Expose Button here
  
    if(keys.available()){
        char ch=keys.readAscii();
        if(isspace(ch))
            return;
        switch(ch){
        case 'A':
            // edit text
            changeState(ST_EDIT_TEXT);
            break;
        case 'B':
            // edit EV
            changeState(ST_EDIT_EV);
            break;
        case 'C':
        case 'D':
            changeState(ST_MAIN);
            break;
        case '#':
        case '*':
            exec.setProgram(&current);
            changeState(ST_EXEC);
            break;
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
            // change to different step for editing
            expnum=ch-'1';
            current[expnum].display(disp, dispbuf, false);
            break;
        default:
            errorBeep();
        }
    }
}

void FstopTimer::st_edit_ev_enter()
{
    deckey.setContext(&expctx);
}

void FstopTimer::st_edit_ev_poll()
{
    if(deckey.poll()){
        if(expctx.exitcode != Keypad::KP_C){
            current[expnum].stops=expctx.result;
        }
        current[expnum].display(disp, dispbuf, false);
        curstate=ST_EDIT;
    }
}

void FstopTimer::st_edit_text_enter()
{
    keys.setContext(&smsctx);
}

void FstopTimer::st_edit_text_poll()
{
    if(keys.poll()){
        if(smsctx.exitcode != Keypad::KP_C){
            strcpy(current[expnum].text, smsctx.buffer);
        }
        current[expnum].display(disp, dispbuf, false);
        curstate=ST_EDIT;        
    }
}

void FstopTimer::st_io_enter()
{
    disp.clear();
    disp.print("A: New  B: Load");
    disp.setCursor(0, 1);
    disp.print("C: Save D: Main");
}

void FstopTimer::st_io_poll()
{
    if(keys.available()){
        char ch=keys.readAscii();
        switch(ch){
        case 'A':
            current.clear();
            changeState(ST_EDIT);
            break;
        case 'B':
            changeState(ST_IO_LOAD);
            break;
        case 'C':
            changeState(ST_IO_SAVE);
            break;
        case 'D':
            changeState(ST_MAIN);
            break;
        default:
            errorBeep();
        }
    }
}

void FstopTimer::st_io_load_enter()
{
    disp.clear();
    disp.print("Select Load Slot");
    deckey.setContext(&intctx);
}

void FstopTimer::st_io_load_poll()
{
    if(deckey.poll()){
        disp.clear();
        char slot=intctx.result;
        if(slot >= Program::FIRSTSLOT && slot <= Program::LASTSLOT){
            current.load(slot);
            disp.print("Program Loaded");
        }
        else{
            disp.print("Slot not in 1..7");
            errorBeep();
        }
        delay(1000);

        changeState(ST_EDIT);
    }
}

void FstopTimer::st_io_save_enter()
{
    disp.clear();
    disp.print("Select Save Slot");
    deckey.setContext(&intctx);
}

void FstopTimer::st_io_save_poll()
{
    if(deckey.poll()){
        disp.clear();
        char slot=intctx.result;
        if(slot >= Program::FIRSTSLOT && slot <= Program::LASTSLOT){
            current.save(slot);
            disp.print("Program Saved");
        }
        else{
            disp.print("Slot not in 1..7");
            errorBeep();
        }
        delay(1000);

        changeState(ST_MAIN);
    }
}

void FstopTimer::st_test_enter()
{
    disp.clear();
    disp.print("A:");
    disp.print(stripcover ? "Cover" : "Indiv");
    disp.print(" B:Change");
    disp.setCursor(0, 1);
    dtostrf(0.01f*stripbase, 0, 2, dispbuf);
    disp.print(dispbuf);
    disp.print(" by ");
    dtostrf(0.01f*stripstep, 0, 2, dispbuf);
    disp.print(dispbuf);
    disp.setCursor(15, 1);
    disp.print(drydown_apply ? "D" : " ");
}

void FstopTimer::st_test_poll()
{
    if(keys.available()){
        char ch=keys.readAscii();
        switch(ch){
        case 'A':
            // toggle type
            stripcover=!stripcover;
            EEPROM.write(EE_STRIPCOV, stripcover);
            changeState(ST_TEST);
            break;
        case 'B':
            // change exposures
            changeState(ST_TEST_CHANGEB);
            break;
        case 'C':
            // main menu
            changeState(ST_MAIN);
            break;
        case 'D':
            // toggle drydown
            toggleDrydown();
            changeState(ST_TEST);
            break;
        case '#':
            // perform exposure
            strip.configureStrip(stripbase, stripstep, stripcover);
            exec.setProgram(&strip);
            changeState(ST_EXEC);
            break;
        }
    }
}

void FstopTimer::st_test_changeb_enter()
{
    disp.clear();
    disp.print("Test Strip Base:");
    deckey.setContext(&expctx);
}

void FstopTimer::st_test_changeb_poll()
{
    if(deckey.poll()){
        if(expctx.exitcode != Keypad::KP_C){
            stripbase=expctx.result;
            EEPROM.write(EE_STRIPBASE, (stripbase >> 8) & 0xFF);
            EEPROM.write(EE_STRIPBASE+1, stripbase & 0xFF);
        }
        changeState(ST_TEST_CHANGES);
    }
}

void FstopTimer::st_test_changes_enter()
{
    disp.clear();
    disp.print("Test Strip Step:");
    deckey.setContext(&stepctx);
}

void FstopTimer::st_test_changes_poll()
{
    if(deckey.poll()){
        if(stepctx.exitcode != Keypad::KP_C){
            stripstep=stepctx.result;
            EEPROM.write(EE_STRIPSTEP, (stripstep >> 8) & 0xFF);
            EEPROM.write(EE_STRIPSTEP+1, stripstep & 0xFF);
        }
        changeState(ST_TEST);
    }
}

void FstopTimer::st_config_enter()
{
    disp.clear();
    disp.setCursor(0,0);
    disp.print("Configuration");
    disp.setCursor(0,1);
    disp.print("B:Brite D:Drydn");  
}

void FstopTimer::st_config_poll()
{
    if(keys.available()){
        char ch=keys.readAscii();
        switch(ch){
        case 'B':
            // change brightness
            ++brightness;
            if(brightness > BL_MAX)
                brightness=BL_MIN;
            analogWrite(pin_backlight, (1<<brightness)-1);
            EEPROM.write(EE_BACKLIGHT, brightness);
            break;
        case 'D':
            // change drydown
            changeState(ST_CONFIG_DRY);
            break;
        default:
            // main menu
            changeState(ST_MAIN);
        }
    }
}

void FstopTimer::st_config_dry_enter()
{
    disp.clear();
    disp.print("Drydown Factor:");
    deckey.setContext(&dryctx);
}

void FstopTimer::st_config_dry_poll()
{
    if(deckey.poll()){
        disp.clear();
        if(dryctx.exitcode == Keypad::KP_C){
            disp.print("Cancelled");
        }
        else{
            drydown=abs(dryctx.result);
            EEPROM.write(EE_DRYDOWN, drydown);
            disp.print("Changed");
        }
        disp.setCursor(0, 1);
        disp.print("Drydown = ");
        dtostrf(0.01f*drydown, 0, 2, dispbuf);
        disp.print(dispbuf);
        delay(1000);
    
        changeState(ST_MAIN);
    }
}



void FstopTimer::changeState(int st)
{
    prevstate=curstate;
    curstate=st;
    (this->*sm_enter[curstate])();
}


void FstopTimer::poll()
{
    // scan keypad
    keys.scan();
  
    // attend to whatever the state requires
    (this->*sm_poll[curstate])(); 
}



void Program::clear()
{
    // base
    exposures[0].stops=200;
    strcpy(exposures[0].text, "Base Exposure");
    isstrip=false;
   
    // invalid
    for(int i=1;i<MAXEXP;++i){
        exposures[i].stops=0;
        strcpy(exposures[i].text, "Undefined");
    }
}

void Program::configureStrip(int base, int step, bool cov)
{
    isstrip=true;
    cover=cov;

    int expos=base;
    for(char i=0;i<MAXEXP;++i){
        exposures[i].stops=expos;
        strcpy(exposures[i].text, "Strip ");
        dtostrf(0.01f*expos, 1, 2, &exposures[i].text[6]);
        strcpy(&exposures[i].text[10], cov ? " Cov" : " Ind");

        expos+=step;
    }
}

void Program::clipExposures()
{
    // don't want to be here forever or overflow the screen
    for(char i=0;i<MAXEXP;++i){
        if(exposures[i].ms > MAXMS) 
            exposures[i].ms=MAXMS;
    }
}

bool Program::compile(char dryval)
{
    if(isstrip){
        if(cover)
            compileStripCover(dryval);
        else
            compileStripIndiv(dryval);

        clipExposures();
        return true;
    }
    else{
        return compileNormal(dryval);
    }
}

void Program::compileStripIndiv(char dryval)
{
    for(int i=0;i<MAXEXP;++i){
        exposures[i].ms=hunToMillis(exposures[i].stops-dryval);
    }
}

void Program::compileStripCover(char dryval)
{
    unsigned long sofar=0;

    for(int i=0;i<MAXEXP;++i){
        unsigned long thisexp=hunToMillis(exposures[i].stops-dryval);
        exposures[i].ms=thisexp-sofar;
        sofar+=exposures[i].ms;
    }
}

bool Program::compileNormal(char dryval)
{
    // base exposure, perhaps with drydown
    int base=exposures[0].stops-dryval;
      
    exposures[0].ms=hunToMillis(base);
    
    // figure out the total exposure desired for each step (base+adjustment)
    unsigned long dodgetime=0;
    for(int i=1;i<MAXEXP;++i){
        if(exposures[i].stops == 0){
            exposures[i].ms=0;
            continue;
        }

        exposures[i].ms=hunToMillis(base+exposures[i].stops);


        // is dodge?
        if(exposures[i].stops < 0){
            // keep track of total time spent dodging          
            dodgetime+=exposures[0].ms-exposures[i].ms;  
            // convert to time-difference
            exposures[i].ms=exposures[0].ms-exposures[i].ms;
        }
        else{
            // convert to time-difference
            exposures[i].ms-=exposures[0].ms;          
        }
    }
    
    // fail if we have more dodge than base exposure
    if(dodgetime > exposures[0].ms)
        return false;
    
    // take off the dodgetime
    exposures[0].ms-=dodgetime;
    
    clipExposures();

    return true;
}

void Program::Exposure::display(LiquidCrystal &disp, char *buf, bool lin)
{
    // print text
    disp.clear();
    disp.setCursor(0,0);
    disp.print(text);
   
    displayTime(disp, buf, lin);
}

void Program::Exposure::displayTime(LiquidCrystal &disp, char *buf, bool lin)
{
    disp.setCursor(0,1);

    // print stops
    char used=0;
    if(stops >= 0){
        disp.print("+");
        ++used;
    }
    dtostrf(0.01f*stops, 0, 2, buf);
    used+=strlen(buf);
    disp.print(buf);

    // print compiled seconds
    if(lin){
        disp.print("=");
        dtostrf(0.001f*ms, 0, 3, buf);
        used+=strlen(buf)+2;
        disp.print(buf);
        disp.print("s");
     
        // fill out to 15 chars with spaces
        // keep the 16th for 'D' drydown-indicator
        for(int i=0;i<15-used;++i)
            buf[i]=' ';
        buf[15-used]='\0';
        disp.print(buf);
    }  
}

Program::Exposure &Program::operator[](int which)
{
    return exposures[which]; 
}

unsigned long Program::hunToMillis(int hunst)
{
    return lrint(1000.0f*pow(2.0f, 0.01f*hunst));
}

void Program::save(int slot)
{
    if(slot < 1 || slot > 7)
        return;
    
    int addr=SLOTBASE+(slot<<SLOTBITS);
    for(int i=0;i<MAXEXP;++i){
        EEPROM.write(addr++, (exposures[i].stops >> 8) & 0xFF);
        EEPROM.write(addr++, exposures[i].stops & 0xFF);
        for(int t=0;t<TEXTLEN;++t){
            EEPROM.write(addr++, exposures[i].text[t]); 
        }
    }
}

void Program::load(int slot)
{
    if(slot < 1 || slot > 7)
        return;
    
    int addr=SLOTBASE+(slot<<SLOTBITS);
    int tmp;
    for(int i=0;i<MAXEXP;++i){
        tmp=EEPROM.read(addr++) << 8;
        tmp|=EEPROM.read(addr++);
        exposures[i].stops=tmp;
        for(int t=0;t<TEXTLEN;++t){
            exposures[i].text[t]=EEPROM.read(addr++); 
        }
     
        // terminate string in memory
        exposures[i].text[TEXTLEN]='\0';
    }    
}


void FstopTimer::errorBeep()
{
    tone(pin_beep, 600, 50); 
}

Executor::Executor(LiquidCrystal &l, Keypad &k, char p_e)
    : disp(l), keys(k), pin_expose(p_e)
{
    current=NULL;
}

void Executor::begin()
{
    execphase=0;
    dd=false;
}

void Executor::setProgram(Program *p)
{
    current=p;
    changePhase(0);
}


void Executor::setDrydown(bool d)
{
    dd=d;
    changePhase(0);
}

/// specify that program is up to a particular exposure; display it
void Executor::changePhase(unsigned char ph)
{
    execphase=ph;

    if(NULL == current)
        return;

    (*current)[ph].display(disp, dispbuf, true);
    disp.setCursor(15, 1);
    disp.print(dd ? "D" : " ");
}

/// perform a measured exposure, updating display with time remaining
/// normal input processing suspended here for a busy-loop, we're only
/// looking at time and the Expose-button for cancellation
void Executor::expose()
{
    if(NULL == current)
        return;

    // backup the duration; it will get overwritten for display purposes
    Program::Exposure &expo=(*current)[execphase];
    unsigned long msbackup=expo.ms;
    unsigned long start=micros();
    unsigned long now=start, dt=0, lastupdate=start;

    // begin
    digitalWrite(pin_expose, HIGH);
  
    // polling loop
    bool cancelled=false, skipped=false;
    while(!skipped){
        now=micros();
        dt=(now-start)/1000;    // works with overflows

        // time is elapsed
        if(dt >= msbackup){
            break; 
        }
    
        // time for some IO?  otherwise tighten the loop a bit
        if((msbackup-dt) > 50){
            if((now-lastupdate) > 100000){
                // re-display with reduced time remaining
                expo.ms=msbackup-dt; 
                expo.displayTime(disp, dispbuf, true);
                lastupdate=now;
            }

            // pause!
            keys.scan();
            if(keys.available()){
                if(keys.readRaw() == Keypad::KP_HASH){
                    digitalWrite(pin_expose, LOW);
                    unsigned long pausestart=micros();

                    // cancel on anuthing but Expose button
                    char ch=keys.readBlocking();
                    switch(ch){
                    case Keypad::KP_HASH:
                        // adjust clock and resume exposing
                        // !! should account for enlarger warmup here
                        start+=micros()-pausestart;
                        digitalWrite(pin_expose, HIGH);
                        break;
                    case Keypad::KP_B:
                        // halt and this exposure
                        skipped=true;
                        break;
                    default:
                        // halt and cancel
                        skipped=true;
                        cancelled=true;
                    }
                }
            }
        }
    }
  
    // cease
    digitalWrite(pin_expose, LOW);
  
    // restore
    expo.ms=msbackup;
  
    if(cancelled){
        // tell user to go home
        disp.clear();
        disp.print("Prog Cancelled");
        delay(1000);
        changePhase(0);
    }   
    else{
        nextPhase();
    }
}

void Executor::nextPhase()
{
    // decide on next exposure or reset to beginning
    for(int newphase=execphase+1;newphase<Program::MAXEXP;++newphase){
        if((*current)[newphase].stops != 0){
            changePhase(newphase);
            return;
        }
    }

    disp.clear();
    disp.print("Program Complete");
    delay(1000);
    changePhase(0);
    
}
