/*
    Copyright (C) 2011 William Brodie-Tyrrell
    william@brodie-tyrrell.org
  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of   
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>
*/


#include <LiquidCrystal.h>
#include <EEPROM.h>
#include <Keypad.h>
#include <FstopTimer.h>

/// functions to exec on entering each state
FstopTimer::voidfunc FstopTimer::sm_enter[]
   ={ &FstopTimer::st_main_enter,
      &FstopTimer::st_edit_enter, 
      &FstopTimer::st_edit_ev_enter, 
      &FstopTimer::st_edit_text_enter, 
      &FstopTimer::st_exec_enter,
      &FstopTimer::st_focus_enter,
      &FstopTimer::st_io_enter, 
      &FstopTimer::st_io_load_enter, 
      &FstopTimer::st_io_save_enter, 
      &FstopTimer::st_test_enter, 
      &FstopTimer::st_config_enter,
      &FstopTimer::st_config_dry_enter };
/// functions to exec when polling within each state
FstopTimer::voidfunc FstopTimer::sm_poll[]
   ={ &FstopTimer::st_main_poll, 
      &FstopTimer::st_edit_poll,
      &FstopTimer::st_edit_ev_poll,
      &FstopTimer::st_edit_text_poll,
      &FstopTimer::st_exec_poll, 
      &FstopTimer::st_focus_poll, 
      &FstopTimer::st_io_poll, 
      &FstopTimer::st_io_load_poll, 
      &FstopTimer::st_io_save_poll, 
      &FstopTimer::st_test_poll,
      &FstopTimer::st_config_poll,
      &FstopTimer::st_config_dry_poll };


FstopTimer::FstopTimer(LiquidCrystal &l, SMSKeypad &k, 
                       char p_e, char p_eb, char p_b, char p_bl)
    : disp(l), keys(k), 
      smsctx(&inbuf[0], 14, &disp, 0, 0),
      deckey(keys),
      expctx(&inbuf[0], 1, 2, &disp, 0, 1, true),
      dryctx(&inbuf[0], 0, 2, &disp, 0, 1, false),
      intctx(&inbuf[0], 1, 0, &disp, 0, 1, false),
      exec(l, keys, p_e),
      pin_expose(p_e), pin_exposebtn(p_eb),
      pin_beep(p_b), pin_backlight(p_bl)
{

    // init libraries
    prevstate=curstate=ST_MAIN;
}

void FstopTimer::begin()
{
    // init raw IO
    pinMode(pin_expose, OUTPUT);
    pinMode(pin_beep, OUTPUT);
    pinMode(pin_backlight, OUTPUT);
    pinMode(pin_exposebtn, INPUT);

    digitalWrite(pin_expose, LOW);

    // load & apply backlight settings
    brightness=EEPROM.read(EE_BACKLIGHT);
    if(brightness > BL_MAX)
        brightness=BL_MAX;
    if(brightness < BL_MIN)
        brightness=BL_MIN;
    analogWrite(pin_backlight, (1<<brightness)-1);
   
    disp.clear();
    disp.print("F/Stop Timer 0.1");
    disp.setCursor(0, 1);
    disp.print("W Brodie-Tyrrell");

    // wait
    keys.readBlocking();

    // boot the state machine
    changeState(ST_MAIN);

    exec.begin();
}

/// begin the main-menu state; write to display
void FstopTimer::st_main_enter()
{
    disp.clear();
    disp.setCursor(0,0);
    disp.print("A:Edit   B: IO");
    disp.setCursor(0,1);
    disp.print("C:Config D: Test");
}

/// polling function executed while in the Main state
void FstopTimer::st_main_poll()
{   
    if(keys.available()){
        char ch=keys.readAscii();
        if(isspace(ch))
            return;
        switch(ch){
        case 'A':
            // exec.current.clear();
            changeState(ST_EDIT);
            break;
        case 'B':
            changeState(ST_IO);        
            break;
        case 'C':
            changeState(ST_CONFIG);
            break;
        case 'D':
            changeState(ST_TEST);
            break;
        case '#':
            changeState(ST_EXEC);
            break;
        case '*':
            changeState(ST_FOCUS);
            break;
        case '$':
            changeState(ST_EXEC);
            break;
        default:
            errorBeep();
        }
    }
}


void FstopTimer::st_exec_enter()
{
    disp.clear();
 
    if(!exec.current.compile(exec.isDrydown(), exec.getDrydown())){
        disp.print("Cannot Print");
        disp.setCursor(0, 1);
        disp.print("Dodges > Base");
        errorBeep();
        delay(2000);
        changeState(ST_EDIT);
    }
    else{
        exec.changePhase(0);
    }
}

void FstopTimer::st_exec_poll()
{
    if(keys.available()){
        char ch=keys.readAscii();
        if(isspace(ch))
            return;
        switch(ch){
        case '#':
            // perform exposure!
            exec.expose();
            break;
        case '*':
            // focus
            changeState(ST_FOCUS);
            break;
        case 'D':
            // toggle drydown & recompile
            exec.toggleDrydown();
            if(exec.getPhase() != 0){
                disp.clear();
                disp.print("Restart Exposure");
                disp.setCursor(0,1);
                disp.print("For Drydown Chg");
                delay(1000);
            }
            // forces recompile -> apply drydown
            changeState(ST_EXEC);
            break;
        case 'A':
            // abort/restart
            disp.clear();
            disp.print("Restart Exposure");
            delay(1000);
            exec.changePhase(0);
            break;
        case 'C':
            // main menu
            changeState(ST_MAIN);
            break;
        } 
    }
}

void FstopTimer::st_focus_enter()
{
    disp.clear();
    disp.print("     Focus!");    
    digitalWrite(pin_expose, HIGH);
}

void FstopTimer::st_focus_poll()
{
    // return to prev state
    if(keys.available()){
        keys.readRaw();
        digitalWrite(pin_expose, LOW);
        changeState(prevstate);
    }
}

void FstopTimer::st_edit_enter()
{
    // don't print help
/*
    disp.clear();
    disp.setCursor(0,0);
    disp.print("Edit Program");
    disp.setCursor(0,1);
    disp.print("A:TX B:EV n:SLOT");
    delay(1000);
*/

    expnum=0;
    exec.current[expnum].display(disp, dispbuf, false);
}

void FstopTimer::st_edit_poll()
{
    // also check for the Expose Button here
  
    if(keys.available()){
        char ch=keys.readAscii();
        if(isspace(ch))
            return;
        switch(ch){
        case 'A':
            // edit text
            changeState(ST_EDIT_TEXT);
            break;
        case 'B':
            // edit EV
            changeState(ST_EDIT_EV);
            break;
        case 'C':
        case 'D':
            changeState(ST_MAIN);
            break;
        case '#':
        case '*':
            changeState(ST_EXEC);
            break;
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
            // change to different step for editing
            expnum=ch-'1';
            exec.current[expnum].display(disp, dispbuf, false);
            break;
        default:
            errorBeep();
        }
    }
}

void FstopTimer::st_edit_ev_enter()
{
    deckey.setContext(&expctx);
}

void FstopTimer::st_edit_ev_poll()
{
    if(deckey.poll()){
        if(expctx.exitcode != Keypad::KP_C){
            exec.current[expnum].stops=expctx.result;
        }
        exec.current[expnum].display(disp, dispbuf, false);
        curstate=ST_EDIT;
    }
}

void FstopTimer::st_edit_text_enter()
{
    keys.setContext(&smsctx);
}

void FstopTimer::st_edit_text_poll()
{
    if(keys.poll()){
        if(smsctx.exitcode != Keypad::KP_C){
            strcpy(exec.current[expnum].text, smsctx.buffer);
        }
        exec.current[expnum].display(disp, dispbuf, false);
        curstate=ST_EDIT;        
    }
}

void FstopTimer::st_io_enter()
{
    disp.clear();
    disp.print("A: New  B: Load");
    disp.setCursor(0, 1);
    disp.print("C: Save D: Main");
}

void FstopTimer::st_io_poll()
{
    if(keys.available()){
        char ch=keys.readAscii();
        switch(ch){
        case 'A':
            exec.current.clear();
            changeState(ST_EDIT);
            break;
        case 'B':
            changeState(ST_IO_LOAD);
            break;
        case 'C':
            changeState(ST_IO_SAVE);
            break;
        case 'D':
            changeState(ST_MAIN);
            break;
        default:
            errorBeep();
        }
    }
}

void FstopTimer::st_io_load_enter()
{
    disp.clear();
    disp.print("Select Load Slot");
    deckey.setContext(&intctx);
}

void FstopTimer::st_io_load_poll()
{
    if(deckey.poll()){
        disp.clear();
        char slot=intctx.result;
        if(slot >= Program::FIRSTSLOT && slot <= Program::LASTSLOT){
            exec.current.load(slot);
            disp.print("Program Loaded");
        }
        else{
            disp.print("Slot not in 1..7");
            errorBeep();
        }
        delay(1000);

        changeState(ST_EDIT);
    }
}

void FstopTimer::st_io_save_enter()
{
    disp.clear();
    disp.print("Select Save Slot");
    deckey.setContext(&intctx);
}

void FstopTimer::st_io_save_poll()
{
    if(deckey.poll()){
        disp.clear();
        char slot=intctx.result;
        if(slot >= Program::FIRSTSLOT && slot <= Program::LASTSLOT){
            exec.current.save(slot);
            disp.print("Program Saved");
        }
        else{
            disp.print("Slot not in 1..7");
            errorBeep();
        }
        delay(1000);

        changeState(ST_MAIN);
    }
}

void FstopTimer::st_test_enter()
{
    disp.clear();
    disp.print(" Test Strip Not");
    disp.setCursor(2,1);
    disp.print("Implemented");
    delay(1000);
    changeState(ST_MAIN);
}

void FstopTimer::st_test_poll()
{
}

void FstopTimer::st_config_enter()
{
    disp.clear();
    disp.setCursor(0,0);
    disp.print("Configuration");
    disp.setCursor(0,1);
    disp.print("B:Brite D:Drydn");  
}

void FstopTimer::st_config_poll()
{
    if(keys.available()){
        char ch=keys.readAscii();
        switch(ch){
        case 'B':
            // change brightness
            ++brightness;
            if(brightness > BL_MAX)
                brightness=BL_MIN;
            analogWrite(pin_backlight, (1<<brightness)-1);
            EEPROM.write(EE_BACKLIGHT, brightness);
            break;
        case 'D':
            // change drydown
            changeState(ST_CONFIG_DRY);
            break;
        default:
            // main menu
            changeState(ST_MAIN);
        }
    }
}

void FstopTimer::st_config_dry_enter()
{
    disp.clear();
    disp.print("Drydown Factor:");
    deckey.setContext(&dryctx);
}

void FstopTimer::st_config_dry_poll()
{
    if(deckey.poll()){
        disp.clear();
        if(dryctx.exitcode == Keypad::KP_C){
            disp.print("Cancelled");
        }
        else{
            exec.setDrydown(abs(dryctx.result));
            disp.print("Changed");
        }
        disp.setCursor(0, 1);
        disp.print("Drydown = ");
        dtostrf(0.01f*exec.getDrydown(), 0, 2, dispbuf);
        disp.print(dispbuf);
        delay(1000);
    
        changeState(ST_MAIN);
    }
}



void FstopTimer::changeState(int st)
{
    prevstate=curstate;
    curstate=st;
    (this->*sm_enter[curstate])();
}


void FstopTimer::poll()
{
    // scan keypad
    keys.scan();
  
    // attend to whatever the state requires
    (this->*sm_poll[curstate])(); 
}



void Program::clear()
{
    // base
    exposures[0].stops=200;
    strcpy(exposures[0].text, "Base Exposure");
   
    // invalid
    for(int i=1;i<MAXEXP;++i){
        exposures[i].stops=0;
        strcpy(exposures[i].text, "Undefined");
    }
}


bool Program::compile(bool dd, char dryval)
{
    // base exposure, perhaps with drydown
    int base=exposures[0].stops;
    if(dd)
        base-=dryval;
      
    exposures[0].ms=hunToMillis(base);
    if(exposures[0].ms > MAXMS)
        exposures[0].ms=MAXMS;
    
    // figure out the total exposure desired for each step (base+adjustment)
    unsigned long dodgetime=0;
    for(int i=1;i<MAXEXP && exposures[i].stops != 0;++i){
        exposures[i].ms=hunToMillis(base+exposures[i].stops);

        // don't want to be here forever or overflow the screen
        if(exposures[i].ms > MAXMS)
            exposures[i].ms=MAXMS;

        // is dodge?
        if(exposures[i].stops < 0){
            // keep track of total time spent dodging          
            dodgetime+=exposures[0].ms-exposures[i].ms;  
            // convert to time-difference
            exposures[i].ms=exposures[0].ms-exposures[i].ms;
        }
        else{
            // convert to time-difference
            exposures[i].ms-=exposures[0].ms;          
        }
    }
    
    // fail if we have more dodge than base exposure
    if(dodgetime > exposures[0].ms)
        return false;
    
    // take off the dodgetime
    exposures[0].ms-=dodgetime;
    
    return true;
}

void Program::Exposure::display(LiquidCrystal &disp, char *buf, bool lin)
{
    // print text
    disp.clear();
    disp.setCursor(0,0);
    disp.print(text);
   
    displayTime(disp, buf, lin);
}

void Program::Exposure::displayTime(LiquidCrystal &disp, char *buf, bool lin)
{
    disp.setCursor(0,1);

    // print stops
    char used=0;
    if(stops >= 0){
        disp.print("+");
        ++used;
    }
    dtostrf(0.01f*stops, 0, 2, buf);
    used+=strlen(buf);
    disp.print(buf);

    // print compiled seconds
    if(lin){
        disp.print("=");
        dtostrf(0.001f*ms, 0, 3, buf);
        used+=strlen(buf)+2;
        disp.print(buf);
        disp.print("s");
     
        // fill out to 15 chars with spaces
        // keep the 16th for 'D' drydown-indicator
        for(int i=0;i<15-used;++i)
            buf[i]=' ';
        buf[15-used]='\0';
        disp.print(buf);
    }  
}

Program::Exposure &Program::operator[](int which)
{
    return exposures[which]; 
}

unsigned long Program::hunToMillis(int hunst)
{
    return lrint(1000.0f*pow(2.0f, 0.01f*hunst));
}

void Program::save(int slot)
{
    if(slot < 1 || slot > 7)
        return;
    
    int addr=SLOTBASE+(slot<<SLOTBITS);
    for(int i=0;i<MAXEXP;++i){
        EEPROM.write(addr++, (exposures[i].stops >> 8) & 0xFF);
        EEPROM.write(addr++, exposures[i].stops & 0xFF);
        for(int t=0;t<TEXTLEN;++t){
            EEPROM.write(addr++, exposures[i].text[t]); 
        }
    }
}

void Program::load(int slot)
{
    if(slot < 1 || slot > 7)
        return;
    
    int addr=SLOTBASE+(slot<<SLOTBITS);
    int tmp;
    for(int i=0;i<MAXEXP;++i){
        tmp=EEPROM.read(addr++) << 8;
        tmp|=EEPROM.read(addr++);
        exposures[i].stops=tmp;
        for(int t=0;t<TEXTLEN;++t){
            exposures[i].text[t]=EEPROM.read(addr++); 
        }
     
        // terminate string in memory
        exposures[i].text[TEXTLEN]='\0';
    }    
}


void FstopTimer::errorBeep()
{
    tone(pin_beep, 600, 50); 
}

Executor::Executor(LiquidCrystal &l, Keypad &k, char p_e)
    : disp(l), keys(k), pin_expose(p_e)
{
}

void Executor::begin()
{
    execphase=0;
    current.clear();
    drydown=EEPROM.read(EE_DRYDOWN);
    drydown_apply=EEPROM.read(EE_DRYAPPLY);
}

/// specify that program is up to a particular exposure; display it
void Executor::changePhase(unsigned char ph)
{
    execphase=ph;
    current[ph].display(disp, dispbuf, true);
    disp.setCursor(15, 1);
    disp.print(drydown_apply ? "D" : " ");
}

/// perform a measured exposure, updating display with time remaining
/// normal input processing suspended here for a busy-loop, we're only
/// looking at time and the Expose-button for cancellation
void Executor::expose()
{
    // backup the duration; it will get overwritten for display purposes
    Program::Exposure &expo=current[execphase];
    unsigned long msbackup=expo.ms;
    unsigned long start=micros();
    unsigned long now=start, dt=0, lastupdate=start;

    // begin
    digitalWrite(pin_expose, HIGH);
  
    // polling loop
    bool cancelled=false;
    while(!cancelled){
        now=micros();
        dt=(now-start)/1000;    // works with overflows

        // time is elapsed
        if(dt >= msbackup){
            break; 
        }
    
        // time for some IO?  otherwise tighten the loop a bit
        if((msbackup-dt) > 50){
            if((now-lastupdate) > 100000){
                // re-display with reduced time remaining
                expo.ms=msbackup-dt; 
                expo.displayTime(disp, dispbuf, true);
                lastupdate=now;
            }

            // pause!
            keys.scan();
            if(keys.available()){
                if(keys.readRaw() == Keypad::KP_HASH){
                    digitalWrite(pin_expose, LOW);
                    unsigned long pausestart=micros();

                    // cancel on anuthing but Expose button
                    if(keys.readBlocking() != Keypad::KP_HASH){
                        cancelled=true;
                    }
                    else{
                        // adjust clock and resume exposing
                        // !! should account for enlarger warmup here
                        start+=micros()-pausestart;
                        digitalWrite(pin_expose, HIGH);
                    }
                }
            }
        }
    }
  
    // cease
    digitalWrite(pin_expose, LOW);
  
    // restore
    expo.ms=msbackup;
  
    if(cancelled){
        // tell user to go home
        disp.clear();
        disp.print("Program Cancelled");
        delay(1000);
        changePhase(0);
    }   
    else{
        // decide on next exposure or reset to beginning
        for(int newphase=execphase+1;newphase<Program::MAXEXP;++newphase){
            if(current[newphase].stops != 0){
                changePhase(newphase);
                return;
            }
        }

        disp.clear();
        disp.print("Program Complete");
        delay(1000);
        changePhase(0);
    }
}

void Executor::setDrydown(char dd)
{
    drydown=dd;
    EEPROM.write(EE_DRYDOWN, drydown);
}
